﻿using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using System.Linq.Expressions;
using System.Security.Cryptography;
using System.Threading.Tasks;
using UnityEngine;
using UnityEngine.Tilemaps;

//GameManager is a singleton class responsible for managing most control flow aspects of a single game instance
public class GameManager : MonoBehaviour
{
    //Game properties
    public GameObject UICanvasGameObject;
    public GameObject MessageBoxUIContentObject;
    public GameObject GameObjects;
    public GameObject MinimapGameObject;
    public GameObject MissionsUIGameObject;
    public MissionResultsOverlayController MissionResultsOverlayController;
    public GameObject PlayerCarGameObject;
    public GameObject TilemapGridGameObject;
    public DebugMenu DebugMenu;

    //UI menus
    public CarSelector CarSelector;
    public MissionInfoMenu MissionInfoMenu { get; private set; }

    //Tilemap layers
    public Tilemap TerrainTilemap { get; private set; }
    public Tilemap RoadsTilemap { get; private set; }
    public Tilemap BuildingsTilemap { get; private set; }
    public Tilemap DecorationTilemap { get; private set; }
    public Tilemap CollisionTilemap { get; private set; }
    public Tilemap MinimapTilemap { get; private set; }

    public int Money { get; private set; }
    public AudioFile CurrentMusicFile { get; set; }

    public Constants.GameState GameState { get; private set; } = Constants.GameState.Playing;

    //public bool IsPaused { get; set; }
    //public bool CanPause { get; set; } = true;
    //public Constants.InGameMenus OpenedMenu = Constants.InGameMenus.None;

    private AudioFile _AmbientMusicFile;

    #region Events

    #region Game State Changed
    public class GameStateChangedEventArgs : EventArgs
    {
        public Constants.GameState OldState { get; private set; }
        public Constants.GameState NewState { get; private set; }

        public GameStateChangedEventArgs(Constants.GameState oldState, Constants.GameState newState)
        {
            OldState = oldState;
            NewState = newState;
        }
    }

    public event EventHandler<GameStateChangedEventArgs> GameStateChanged;

    private void OnGameStateChanged(GameStateChangedEventArgs e)
    {
        EventHandler<GameStateChangedEventArgs> handler = GameStateChanged;
        if(handler != null)
        {
            handler(this, e);
        }
    }
    #endregion

    #region Player Car Changed
    public event EventHandler PlayerCarChanged;

    private void OnPlayerCarChanged()
    {
        EventHandler handler = PlayerCarChanged;

        if(handler != null)
        {
            handler(this, EventArgs.Empty);
        }
    }
    #endregion

    #endregion

    //Singleton
    private static GameManager _Instance;
    public static GameManager Instance
    {
        get
        {
            if (_Instance == null)
            {
                _Instance = FindObjectOfType<GameManager>();
            }

            return _Instance;
        }
    }

    /// <summary>
    /// Starts a new game by performing the generation process and sets everything up
    /// </summary>
    public void StartNewGame()
    {
        //Compute the centre of the city and start there
        Vector3 centre = new Vector3Int(Instance.RoadsTilemap.size.x / 2, Instance.RoadsTilemap.size.y / 2, Constants.TilemapZPosition);
        centre.x += Math.Abs((Constants.GridCellGap.x * centre.x));
        centre.y += Math.Abs((Constants.GridCellGap.y * centre.y));
        Camera.main.transform.position = new Vector3(centre.x, centre.y, ConfigurationManager.Instance.Core.CameraZPosition);
        Camera.main.transform.LookAt(centre);
        
        //Set up the MessageBox, Randomizer, Minimap and starting money
        MessageBoxManager.Instance.SetMessageBoxObject(MessageBoxUIContentObject.FindChild("MessageBox"), MessageBoxUIContentObject);
        Randomizer.Initialize(GameController.Instance.CitySeed);
        MinimapManager.Instance.Initialize(MinimapGameObject, Camera.main);
        Money = ConfigurationManager.Instance.Player.StartingMoney;

        //Diagnostic stopwatches for performance measurements
        System.Diagnostics.Stopwatch fullGenWatch = new System.Diagnostics.Stopwatch();
        System.Diagnostics.Stopwatch passWatch = new System.Diagnostics.Stopwatch();

        fullGenWatch.Start();
        passWatch.Start();

        //Generate Roads with L-Systems
        RoadsPass();
        //Debug.Log("Roads Time: " + passWatch.ElapsedMilliseconds);
        passWatch.Restart();

        //Roads created, let's identify the streets
        StreetsPass();
        //Debug.Log("Streets Time: " + passWatch.ElapsedMilliseconds);
        passWatch.Restart();

        //We have roads and streets, let's place buildings
        BuildingsPass();
        //Debug.Log("Buildings Time: " + passWatch.ElapsedMilliseconds);
        passWatch.Restart();

        //Now let's colour the terrain
        TerrainPass();
        //Debug.Log("Terrain Time: " + passWatch.ElapsedMilliseconds);
        passWatch.Restart();

        //All our key elements are in place, compute collisions
        CollisionPass();
        //Debug.Log("Collision Time: " + passWatch.ElapsedMilliseconds);
        passWatch.Restart();

        //Now we can create the minimap
        MinimapPass();
        //Debug.Log("Minimap Time: " + passWatch.ElapsedMilliseconds);
        passWatch.Restart();

        //And finally generate a selection of cars
        CarsPass(centre);
        //Debug.Log("Cars Time: " + passWatch.ElapsedMilliseconds);

        passWatch.Stop();
        fullGenWatch.Stop();

        //Debug.Log("Full Time: " + fullGenWatch.ElapsedMilliseconds);

        //Set up the music and we're good to play!
        _AmbientMusicFile = AudioManager.Instance.AudioGroups["AmbienceBGMGroup"].GetRandomFile();
        RestartAmbientMusic();
    }

    /// <summary>
    /// Sets the player's current car
    /// </summary>
    /// <param name="playerCarGameObject">The GameObject of the new car</param>
    public void SetPlayerCar(GameObject playerCarGameObject)
    {
        //Store our old data
        Vector3 playerPos = PlayerCarGameObject.transform.position;
        Vector3 playerRot = PlayerCarGameObject.transform.eulerAngles;

        //Destroy the old car
        Destroy(PlayerCarGameObject.gameObject);

        //Create the new one
        PlayerCarGameObject = Instantiate(playerCarGameObject);
        PlayerCarGameObject.name = Constants.PlayerCarObjectName;
        PlayerCarGameObject.transform.parent = GameObjects.transform;
        PlayerCarGameObject.transform.position = playerPos;
        PlayerCarGameObject.transform.eulerAngles = playerRot;
        PlayerCarGameObject.SetActive(true);

        //Reset the icon and raise the event
        MinimapManager.Instance.AddIcon(PlayerCarGameObject, Constants.PlayerCarIconName, ColoursManager.Instance.Colours["MinimapPlayer"].Colour, ConfigurationManager.Instance.Minimap.PlayerIconRenderSize, (int)MinimapIconFlags.MaintainRotation, Resources.Load<Sprite>("Minimap/Icons/minimapPlayerIcon"));
        OnPlayerCarChanged();
    }

    /// <summary>
    /// Stops the current music and restarts the ambient one
    /// </summary>
    public void RestartAmbientMusic()
    {
        AudioManager.Instance.StopFile(CurrentMusicFile);
        CurrentMusicFile = _AmbientMusicFile;
        AudioManager.Instance.PlayFile(_AmbientMusicFile, true);
    }

    /// <summary>
    /// Increments the player's money by the set amount
    /// </summary>
    /// <param name="amount">The amount to incremnt by</param>
    public void IncrementMoney(int amount)
    {
        //Safety checks for caps/boundaries
        if(Money + amount >= ConfigurationManager.Instance.Player.HighMoneyCap)
        {
            Money = ConfigurationManager.Instance.Player.HighMoneyCap;
        }

        else if(Money + amount <= ConfigurationManager.Instance.Player.LowMoneyCap)
        {
            Money = ConfigurationManager.Instance.Player.LowMoneyCap;
        }

        else
        {
            Money += amount;    //Caps and boundaries are fine, apply it
        }
    }

    /// <summary>
    /// Sets the new game state and raises the event
    /// </summary>
    /// <param name="newState">The new state to set</param>
    public void SetGameState(Constants.GameState newState)
    {
        OnGameStateChanged(new GameStateChangedEventArgs(GameState, newState));
        GameState = newState;
    }

    /// <summary>
    /// TerrainPass is responsible for colouring the city terrain
    /// </summary>
    private void TerrainPass()
    {
        //Load our grass tile, get the cells count
        Tile grassTile = Resources.Load("Palette Tiles/Terrain Tiles/grassNew") as Tile;

        int horCellsCount = Convert.ToInt32(Math.Ceiling(RoadsTilemap.size.x / (float)ConfigurationManager.Instance.Terrain.TerrainChunkWidth));
        int verCellsCount = Convert.ToInt32(Math.Ceiling(RoadsTilemap.size.y / (float)ConfigurationManager.Instance.Terrain.TerrainChunkHeight));
        Dictionary<Tuple<int, int>, Color> cellColours = new Dictionary<Tuple<int, int>, Color>();

        //Loop through all the cells and colour them
        for (int hor = 0; hor < horCellsCount; hor++)
        {
            for (int ver = 0; ver < verCellsCount; ver++)
            {
                cellColours[new Tuple<int, int>(hor, ver)] = ColoursManager.Instance.ColourGroups["Grass"].GetRandomColourEntry().Colour;
            }
        }

        //Now identify the noise zone tiles
        Dictionary<Vector3Int, int> noiseZoneTiles = new Dictionary<Vector3Int, int>();

        Vector2Int terrainSize = new Vector2Int
            (
                Convert.ToInt32((Math.Ceiling(BuildingsTilemap.size.x * (1.0f / ConfigurationManager.Instance.Terrain.TerrainChunkWidth)) * ConfigurationManager.Instance.Terrain.TerrainChunkWidth)),
                Convert.ToInt32((Math.Ceiling(BuildingsTilemap.size.y * (1.0f / ConfigurationManager.Instance.Terrain.TerrainChunkHeight)) * ConfigurationManager.Instance.Terrain.TerrainChunkHeight))
                );

        float increment = (float)ConfigurationManager.Instance.Terrain.TerrainChunkWidth;

        //Start at the bottom of the city
        int startingY = GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y > 0 ? 
            Convert.ToInt32(Math.Floor(GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y / increment) * increment) : 
            Convert.ToInt32(Math.Ceiling(Math.Abs(GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y) / increment) * -increment);

        //Loop through the whole terrain
        for (int x = 0; x < terrainSize.x; x++)
        {
            for (int y = startingY; y < terrainSize.y; y++)
            {
                //Get the tile's position, cell and local cell position
                Vector3Int thisTilePos = new Vector3Int(x, y, Constants.TilemapZPosition);
                int thisTileHorCell = Math.Abs(thisTilePos.x / ConfigurationManager.Instance.Terrain.TerrainChunkWidth);
                int thisTileVerCell = Math.Abs(thisTilePos.y / ConfigurationManager.Instance.Terrain.TerrainChunkHeight);
                int cellPosX = Math.Abs(thisTilePos.x % ConfigurationManager.Instance.Terrain.TerrainChunkWidth);
                int cellPosY = Math.Abs(thisTilePos.y % ConfigurationManager.Instance.Terrain.TerrainChunkHeight);

                TerrainTilemap.SetTile(thisTilePos, grassTile);
                TerrainTilemap.SetTileFlags(thisTilePos, TileFlags.None);
                TerrainTilemap.SetColor(thisTilePos, cellColours[new Tuple<int, int>(thisTileHorCell, thisTileVerCell)]);

                bool isInNoiseZone = false;

                if (!noiseZoneTiles.ContainsKey(thisTilePos))
                {
                    if (cellPosX < ConfigurationManager.Instance.Terrain.TerrainNoiseZoneWidth) //Left Edge
                    {
                        isInNoiseZone = true;
                    }

                    else if (ConfigurationManager.Instance.Terrain.TerrainChunkWidth - cellPosX < ConfigurationManager.Instance.Terrain.TerrainNoiseZoneWidth)  //Right Edge
                    {
                        isInNoiseZone = true;
                    }

                    if (cellPosY < ConfigurationManager.Instance.Terrain.TerrainNoiseZoneHeight)    //Bottom Edge
                    {
                        isInNoiseZone = true;
                    }

                    else if (ConfigurationManager.Instance.Terrain.TerrainChunkHeight - cellPosY < ConfigurationManager.Instance.Terrain.TerrainNoiseZoneHeight)    //Top Edge
                    {
                        isInNoiseZone = true;
                    }

                    if (isInNoiseZone)
                    {
                        noiseZoneTiles[thisTilePos] = 0;
                    }
                }
            }
        }

        //UNUSED: Attempt to bilinear interpolate to produce realistic terrain transitions
        if (ConfigurationManager.Instance.Terrain.IsBilinearInterpolationEnabled)
        {
            Dictionary<Vector3Int, Color> finalTiles = new Dictionary<Vector3Int, Color>();

            foreach (KeyValuePair<Vector3Int, int> noiseZonePair in noiseZoneTiles)
            {
                Vector3Int tilePos = noiseZonePair.Key;
                int tileDist = noiseZonePair.Value;

                Color tileColour = TerrainTilemap.GetColor(new Vector3Int(tilePos.x, tilePos.y, tilePos.z));

                Color leftNeighbourColour = tileColour;
                Color rightNeighbourColour = tileColour;
                Color topNeighbourColour = tileColour;
                Color bottomNeighbourColour = tileColour;

                if (tilePos.x > 0)
                {
                    leftNeighbourColour = TerrainTilemap.GetColor(new Vector3Int(tilePos.x - 1, tilePos.y, tilePos.z));
                }

                if (tilePos.x < RoadsTilemap.size.x)
                {
                    rightNeighbourColour = TerrainTilemap.GetColor(new Vector3Int(tilePos.x + 1, tilePos.y, tilePos.z));
                }

                if (tilePos.y > 0)
                {
                    bottomNeighbourColour = TerrainTilemap.GetColor(new Vector3Int(tilePos.x, tilePos.y - 1, tilePos.z));
                }

                if (tilePos.y < RoadsTilemap.size.y)
                {
                    topNeighbourColour = TerrainTilemap.GetColor(new Vector3Int(tilePos.x, tilePos.y + 1, tilePos.z));
                }

                List<Color> colours = new List<Color>() { leftNeighbourColour, rightNeighbourColour, topNeighbourColour, bottomNeighbourColour };
                finalTiles[tilePos] = Utilities.BilinearInterpolate(colours, 0.5f);
            }

            foreach (KeyValuePair<Vector3Int, Color> finalTile in finalTiles)
            {
                TerrainTilemap.SetColor(finalTile.Key, finalTile.Value);
            }
        }
    }

    /// <summary>
    /// RoadsPass utilizies L-Systems to draw the city roads
    /// </summary>
    private void RoadsPass()
    {
        if (!GameController.Instance.LSystem.Draw())
        {
            UnityEngine.Debug.LogError("ERROR: Failed to draw L-System!");
        }
    }

    /// <summary>
    /// StreetsPass uses StreetsManager to identify all streets in the city
    /// </summary>
    private void StreetsPass()
    {
        StreetsManager.Instance.IdentifyStreets();
    }

    /// <summary>
    /// BuildingsPass uses BuildingsManager to place buildings into the city
    /// </summary>
    private async Task BuildingsPass()
    {
        if (!BuildingsManager.Instance.PlaceBuildings())
        {
            BuildingsManager.Instance.Clear();
            await MessageBoxManager.Instance.Show("Buildings Placement Failed!", "Failed to place buildings! The game will still function, but no buildings will be present.");
        }
    }

    /// <summary>
    /// MinimapPass uses MinimapManager to create and initially populate the minimap
    /// </summary>
    private async Task MinimapPass()
    {
        if (ConfigurationManager.Instance.Minimap.IsEnabled)
        {
            MinimapManager.Instance.AddIcon(PlayerCarGameObject, Constants.PlayerCarIconName, ColoursManager.Instance.Colours["MinimapPlayer"].Colour, ConfigurationManager.Instance.Minimap.PlayerIconRenderSize, (int)MinimapIconFlags.MaintainRotation, Resources.Load<Sprite>("Minimap/Icons/minimapPlayerIcon"));

            if (!MinimapManager.Instance.CreateMinimap())
            {
                MinimapGameObject.SetActive(false);
                await MessageBoxManager.Instance.Show("Mini-Map Creation Failed!", "Failed to create mini-map! The game will still function, but the mini-map has been disabled.");
            }
        }

        else
        {
            MinimapGameObject.SetActive(false);
        }
    }

    /// <summary>
    /// CarsPass uses CarsManager to generate a list of cars for each class and set the player's initial car
    /// </summary>
    /// <param name="centre">The centre point of the city</param>
    private void CarsPass(Vector3 centre)
    {
        CarsManager.Instance.RegenerateCars();

        //All generated, let's pick a random low end starting car
        Randomizer.Regenerate();
        int index = Randomizer.RNG.Next(0, CarsManager.Instance.LowEndCars.Count);
        SetPlayerCar(CarsManager.Instance.LowEndCars[index]);

        PlayerCarGameObject.transform.position = new Vector3(centre.x, centre.y, ConfigurationManager.Instance.Player.ZPosition);

        //Make sure we rotate the right way around too
        if (RoadsTilemap.GetTile(Vector3Int.RoundToInt(centre)).name.Contains("horRoad"))
        {
            PlayerCarGameObject.transform.eulerAngles = new Vector3(0.0f, 0.0f, 90.0f);
        }
    }

    /// <summary>
    /// CollisionsPass uses the L-System drawer computed boundaries to set collsions to stop the player from exiting the city
    /// </summary>
    private void CollisionPass()
    {
        EdgeCollider2D collisionEdgeCollider = CollisionTilemap.gameObject.GetComponent<EdgeCollider2D>();
        collisionEdgeCollider.points = new Vector2[]
        {
            new Vector2(GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.x, GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y - 0.5f), //Bottom Left
            new Vector2(GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.x, GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.y + 0.5f), //Top Left
            new Vector2(GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.x, GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.y + 0.5f), //Top Right
            new Vector2(GameController.Instance.LSystem.Drawer.CityBottomRightTilePosition.x, GameController.Instance.LSystem.Drawer.CityBottomRightTilePosition.y - 0.5f), //Bottom Right
            new Vector2(GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.x , GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y - 0.5f), //Bottom Left
        };
    }

    private void Start()
    {
        //Get our UI
        MissionInfoMenu = UICanvasGameObject.FindChild("MissionInfoMenu").GetComponent<MissionInfoMenu>();

        //Get our tilemaps
        TerrainTilemap = TilemapGridGameObject.FindChild("Terrain").GetComponent<Tilemap>();
        RoadsTilemap = TilemapGridGameObject.FindChild("Roads").GetComponent<Tilemap>();
        BuildingsTilemap = TilemapGridGameObject.FindChild("Buildings").GetComponent<Tilemap>();
        DecorationTilemap = TilemapGridGameObject.FindChild("Decoration").GetComponent<Tilemap>();
        CollisionTilemap = TilemapGridGameObject.FindChild("Collision").GetComponent<Tilemap>();
        MinimapTilemap = TilemapGridGameObject.FindChild("Minimap").GetComponent<Tilemap>();

        //Let's start a new game!
        StartNewGame();
    }

    private void LateUpdate()
    {
        Camera.main.transform.position = new Vector3(PlayerCarGameObject.transform.position.x, PlayerCarGameObject.transform.position.y, ConfigurationManager.Instance.Core.CameraZPosition);
        Camera.main.transform.LookAt(PlayerCarGameObject.transform);

        //Left
        if (Camera.main.transform.position.x <= GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.x)
        {
            Camera.main.transform.position = new Vector3(GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.x, Camera.main.transform.position.y, ConfigurationManager.Instance.Core.CameraZPosition);
        }

        //Right
        else if (Camera.main.transform.position.x > GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.x)
        {
            Camera.main.transform.position = new Vector3(GameController.Instance.LSystem.Drawer.CityTopRightTilePosition.x, Camera.main.transform.position.y, ConfigurationManager.Instance.Core.CameraZPosition);
        }

        //Top
        else if (Camera.main.transform.position.y >= GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.y)
        {
            Camera.main.transform.position = new Vector3(Camera.main.transform.position.x, GameController.Instance.LSystem.Drawer.CityTopLeftTilePosition.y, ConfigurationManager.Instance.Core.CameraZPosition);
        }

        //Bottom
        else if (Camera.main.transform.position.y <= GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y)
        {
            Camera.main.transform.position = new Vector3(Camera.main.transform.position.x, GameController.Instance.LSystem.Drawer.CityBottomLeftTilePosition.y, ConfigurationManager.Instance.Core.CameraZPosition);
        }
    }
}
